//
//  MCSourceListManager.h
//  MCApplication
//
//  Created by Michael Clark on Tue Jul 27 2004.
//  Copyright (c) 2004 __MyCompanyName__. All rights reserved.
//

// The source list manager is responsible to managing an active set of source lists.
// It asks a source list host for it's source lists.  The source list host can be any object which
//  implements the MCSourceListHost protocol.
// Once the source list manager gets the source lists it indexes them for quick retrieval (generally source list
//  pane views will use the source list manager as their data source).  It also needs to manager watch lists which
//  require a timed update with the database and notification posting.
// The source list manager also maintains a list of in view source lists (meaning a source list currently open in a window).
// To stay in sync the source list manager observes several notifications for source list selection, addition, deletion 
//  and modification, as well as low level EO notifications for remote changes to shared source lists.

#import <Foundation/Foundation.h>
#import "MCSourceListHost.h"
#import "MCSourceList.h"

// This is an ugly hack - we set this right before we generate a qualifier for a smartlist so that the query processors can check it to alter their output where applicable because they do not have enough context in some situations (this was added for the pipeline/stage criterion in DL).
extern NSString*  gMCSourceListRootEntityName;

@class MCPObjectContext;

@interface MCSourceListManager : NSObject {
	@private
	
	id <MCSourceListHost>    _sourceListHost;
	MCPObjectContext*		 _objectContext;
	id                       _document;
	BOOL                     _sendNotifications;
	
	NSMutableDictionary*     _sourceListCache;
	NSMutableDictionary*     _defaultSourceListCache;
	NSMutableDictionary*     _recentSourceListCache;
	NSMutableDictionary*     _syncSourceListCache;
	NSMutableDictionary*     _touchSourceListCache;
	NSMutableSet*            _watchLists; // Source Lists in the watch lists array are also in the cache!
	NSThread*                _watchListThread;
	NSTimer*                 _watchListTimer; // this should be replaced with the thread once we figure out our thread safety issues
	
	NSMutableDictionary*     _processors;
	NSMutableDictionary*     _customProcessors;
}
//// Class Methods
+ (NSArray *)resultsForSourceLists:(NSArray *)sourceLists; // returns a normalized array of results from the source lists

//// Initializers
- (id)initWithObjectContext: (MCPObjectContext *)anObjectContext document: (id)aDocument sourceListHost: (id <MCSourceListHost>)aHost;

// Notification Processing - called directly on the source list manager (i.e. it does not 
//   observe!!!
- (void)sourceListAdded: (NSNotification*)notif;
- (void)sourceListEdited: (NSNotification*)notif;
- (void)sourceListDeleted: (NSNotification*)notif;
- (BOOL)sendNotifications;
- (void)setSendNotifications:(BOOL)flag;


// Recents management
- (id <MCSourceList>)recentSourceListForSourceListType: (NSString*)type;
- (id <MCSourceList>)createRecentSourceListForSourceListType: (NSString*)type entityName: (NSString*)entityName;

// Sync List management
- (id <MCSourceList>)syncSourceListForSourceListType: (NSString*)type;
- (id <MCSourceList>)createSyncSourceListForSourceListType: (NSString*)type entityName: (NSString*)entityName;

// Touch List Management
- (id <MCSourceList>)touchSourceListForSourceListType: (NSString*)type;
- (id <MCSourceList>)touchSourceListForSourceListType:(NSString *)type host:(id <MCSourceListHost>)aHost;
- (id <MCSourceList>)createTouchSourceListForSourceListType: (NSString*)type entityName: (NSString*)entityName;

// Utility Methods
// Check to make sure the host is up to date with default/system source lists
- (void)ensureHostContinuityForVersion: (NSNumber*)version;
- (void)ensureContinuityForHost: (id <MCSourceListHost>)aHost version: (NSNumber*)version;

- (void)addSourceListsFromFileAtPath: (NSString*)path toHost: (id <MCSourceListHost>)host;

- (id <MCSourceList>)createSourceListForHost: (id <MCSourceListHost>)host sourceListConfig: (NSDictionary*)sourceListConfig;


- (BDQualifier*)qualifierForQueryDisplayState: (NSDictionary*)displayState;

- (BDQualifier*)qualifierForSourceList: (id <MCSourceList>)aSourceList;
- (BDQualifier*)qualifierForWatchList: (id <MCSourceList>)aWatchList;

// Asks the source list host for a new one and returns it
- (id <MCSourceList>)createAndAddSourceList;

//// Source List Pane View data source methods
- (NSArray *)sourceListsForIdentifier: (NSString *)anIdentifier;

/// Returns the source list matching the given primary key
- (id <MCSourceList>)sourceListForPrimaryKey:(NSNumber *)pKey;
/// Returns the source lists macthing the given primary keys for the groupIdentifier
- (NSArray*)sourceListsForPrimaryKeys: (NSArray*)pKeys andGroupIdentifier: (NSString*)groupIdentifier;

- (id <MCSourceList>)defaultSourceListForGroupIdentifier: (NSString*)groupIdentifier;


- (NSDictionary*)processQualifierConfigForDefaultConfig:(NSDictionary*)default_config defaultValue:(NSDictionary*)defaultValues;

//// Accessors
- (MCPObjectContext*)objectContext;

- (id <MCSourceListHost>)sourceListHost;


- (id)processorNamed: (NSString*)name;

- (id)customProcessorNamed: (NSString*)name;

- (NSMutableDictionary *)processors;
- (void)setProcessors:(NSMutableDictionary *)aProcessors;

- (NSMutableDictionary *)customProcessors;
- (void)setCustomProcessors:(NSMutableDictionary *)aCustomProcessors;

@end
